## Probability of transitioning into housing, and projected first home prices that goes to individual wealth
## Rate of "upsizing"/"downsizing" housing value given you are in housing - saving in housing
## Debt reduction rates


## HILDA variables include:
## hsyr - year purchased current home
## hsvalui - home value today
## hsprice - price of home when purchased


# prelims -----------------------------------------------------------------


rm(list=ls())
gc()


## custom smoother function
source("./R scripts/Data and parameters/0 Custom functions.R")



# Read in HILDA grouped master ----------------------------------------------------

hilda_grouped <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs") %>% 
  select(xwaveid, hhrhid, wavenumber, age_grp, age_grp2, total_inc_qtile, hhwtrp, hhwte, 
         total_inc, txtotp, txtotn, contains("assets"), contains("debt"),
         hsyr, hsprice, hsvalui, hsdebti, hwhmvai, pwhmvai, hspown, primary_residence_share,
         pwtpvi, pwtpdi, total_debt, hsmgi,
         pwopvai, pwopdti,
         parent_age, hgage) %>% 
  filter(wavenumber!=19) ## remove wave 19 as less info - no wealth data or home purchase data though housing value data available



# Set up home purchase data -----------------------------------------------

## get data on ALL home purchase for each person, if they ever part of a home purchase
home_purchases <- hilda_grouped %>% 
  ## proportion of housing value the individual owns --- backed out from indiv wealth analysis - same as primary_residence_share where hwhmvai>0 (with minor rounding error)
  mutate(indiv_home_prop = ifelse(hwhmvai!=0, pwhmvai/hwhmvai, NA)) %>%
  ## given that proportion, what amount of price when home purchased is attributed to each indiv? (eg maybe the household bought a house but you dont own any)
  mutate(psprice = ifelse(hsprice>0, indiv_home_prop*hsprice, NA)) %>%
  ## filter to reports of people who had bought a home (available in wealth years)
  filter(!is.na(psprice) & psprice>0 ) %>% 
  ## if people have reported different purchase prices for the same purchase year across different waves, go with the earliest response (assuming better recall)
  arrange(xwaveid, hsyr, wavenumber) %>% 
  group_by(xwaveid, hsyr) %>% 
  filter(row_number()==1) %>% 
  ungroup %>% 
  ## identify each person's home purchase years
  distinct(xwaveid, hsyr, indiv_home_prop, psprice) 

## within hilda years, create "home purchased this year" identifier (for merging later) and "possible first home purchase" identifier - first purchase recorded by the person
home_purchases_hilda_year <- home_purchases %>% 
  group_by(xwaveid) %>% 
  arrange(xwaveid, hsyr) %>% 
  mutate(home_purchased=1 ,
         first_recorded_home_purchase = ifelse(row_number()==1, 1, 0)) %>% 
  ## keep those where the year they purchased the house is within hilda years for merging
  mutate(wavenumber= hsyr-2000) %>% 
  filter(wavenumber>=1) %>% 
  select(-hsyr)

## for first recorded home purchase, what year was home purchased? for each person
home_purchases_first_record <- home_purchases %>% 
  group_by(xwaveid) %>% 
  arrange(xwaveid, hsyr) %>% 
  filter(row_number()==1) %>% 
  mutate(year_first_recorded_home_purchase = hsyr) %>% 
  select(xwaveid, year_first_recorded_home_purchase)


## merge home purchase years with all hilda data
hilda_home_purchases <- hilda_grouped %>% 
  left_join(home_purchases_hilda_year) %>% 
  left_join(home_purchases_first_record) %>% 
  ## max age group in household
  group_by(wavenumber, hhrhid) %>% 
  mutate(max_hh_age = max(age_grp)) %>% 
  arrange(xwaveid, wavenumber) %>% 
  group_by(xwaveid) %>% 
  ## id if likely owned a home previously based on age in hh and HH hspown variable
  mutate(likely_not_first_home = ifelse(hspown ==1 & (age_grp>="[55,60)" | age_grp==max_hh_age), 1, 0)) %>% 
  fill(likely_not_first_home, .direction="down") %>% 
  ## id if someone is a likely first home purchaser
  ungroup %>% 
  mutate(first_home_purchase = case_when(
    ## not first purchase if age is <15
    age_grp<="[10,15)" ~ 0, ## 0 means still don't own a home
    
    ## not first purchaser if earliest record of purchase is before current year
    (year_first_recorded_home_purchase-2000) < wavenumber         ~ -1, ## -1 means already own a home
    ## likely not first purchase if identified above
    likely_not_first_home==1 ~ -1,
    
    ## not first purchaser if never recorded a home purchase
    is.na(year_first_recorded_home_purchase) ~ 0, ## 0 means still don't own a home
    
    ## assume first purchaser if it is the year of first recorded purchase and you are young (assume it is regardless of previous recorded hh housing value)
    first_recorded_home_purchase==1 & age_grp<="[55,60)" ~ 1,
    ## not yet first purchaser in the years before that first home purchase
    year_first_recorded_home_purchase-2000 > wavenumber  & age_grp<="[55,60)" ~ 0,
    
    ## if ever recorded a home purchase and old - assume you have previously purchased a home
    !is.na(year_first_recorded_home_purchase) & age_grp>="[60,65)" ~ -1
    
  ))





# Probability of transitioning into housing (used in base scenario) ----------------------------

## We see the proportion of people in housing at each age group. 
## What is the implied rate of getting into home ownership to get observed proportions at the next age grp?

hilda_ho_rates <- hilda_grouped %>% 
  filter(wavenumber %in% c(2, 6, 10, 14, 18) ) %>% 
  mutate(homeowner= ifelse(housing_assets>0, 1, 0)) %>% 
  ## remove people who have received bequests - use Cigdem and Whelan vals
  filter(!is.na(parent_age))  %>% 
  ## calc home ownership rate
  group_by(wavenumber, hgage, total_inc_qtile) %>% 
  summarise(ho_rate = sum(homeowner*hhwte)/sum(hhwte),
            ho_rate = ifelse(is.na(ho_rate), 0, ho_rate)) %>% 
  ## average over waves
  group_by(hgage, total_inc_qtile) %>% 
  summarise(ho_rate = mean(ho_rate)) 

## get monotonically increasing smooth rates across ages from 15 to 55
## https://stackoverflow.com/questions/28582145/how-to-fit-a-smooth-line-to-some-points-but-preserve-monotonicity
## assume no homeowners under 15, and no more movements into homeownership from age 55. Note ABS data suggests that 95% of FHB are <55
home_trans_prob <- hilda_ho_rates %>% 
  split(., .$total_inc_qtile) %>% 
  lapply(., function(x) {
    x_age1555 <- x %>% filter(hgage>=10 & hgage<=60)
    smooth_rate <- predict(scam( x_age1555$ho_rate ~ s(x_age1555$hgage, k=9, bs="mpi"))) ## k is a smoothing param. mpi means monotonically increasing
    
    x_age1555 <- data.frame(hgage = x_age1555$hgage, 
                            ho_rate_smooth=smooth_rate) %>% 
      filter(hgage>=15 & hgage<=55)
    
    left_join(x, x_age1555)
  }) %>% 
  rbindlist %>% 
  mutate(ho_rate_smooth = ifelse(ho_rate_smooth<0, 0, ho_rate_smooth)) %>% 
  ## assume no homeowners under 15, and no more movements into homeownership from age 55.
  mutate(ho_rate_smooth = ifelse(hgage<15, 0, ho_rate_smooth)) %>% 
  group_by(total_inc_qtile) %>% 
  fill(ho_rate_smooth, .direction="down") %>% 
  ## what is the implied rate of entry into home ownership to get observed home ownership rates, assuming staying within same inc qtile
  mutate(home_trans_prob = (ho_rate_smooth-lag(ho_rate_smooth)) / (1-lag(ho_rate_smooth))) %>% 
  ## ensure a rate for every age_grp and inc combination
  right_join(., expand_grid(total_inc_qtile = unique(hilda_grouped$total_inc_qtile),
                            hgage = c(0:105))) %>% 
  arrange(total_inc_qtile, hgage) %>% 
  ## create age grp and "mortality year" variable
  ## ie for first year the cohort is at that age group, use year 1 rate. For second year, use year 2 rate, etc
  mutate(age_grp = cut(hgage, breaks=seq(0, 105, 5), right=F, include.lowest=T, ordered_result=T)) %>% 
  group_by(age_grp, total_inc_qtile) %>% 
  mutate(mortality_year = row_number()) %>% 
  ## 0 rates for >=55 and <15
  mutate(home_trans_prob = ifelse(age_grp>="[55,60)" | age_grp<="[10,15)", 0, home_trans_prob)) %>% 
  group_by(total_inc_qtile) %>% 
  fill(home_trans_prob, .direction="down") %>% 
  ## take the 5 year average (effectively equally weighting years in each age group)
  group_by(total_inc_qtile) %>% 
  mutate(home_trans_prob_av5 = ma(home_trans_prob, 5) %>% as.numeric %>% lead(2) ) %>% 
  mutate(home_trans_prob_av5 = ifelse(age_grp>="[55,60)" | age_grp<="[10,15)", 0, home_trans_prob_av5))


# ggplot(home_trans_prob %>% filter(hgage>=15 & hgage<=55)) + geom_col(aes(x=hgage, y=home_trans_prob_av5, fill=as.factor(total_inc_qtile)), position="dodge") 


## save  
qsave(home_trans_prob %>% select(age_grp, mortality_year, total_inc_qtile, home_trans_prob = home_trans_prob_av5),
      "./Input data/housing_transition_probs_aiy.qs")



# Probability of transitioning into housing (used in alt scenario) -------------------------------

## we want to identify year purchased FIRST house.
## every 4 years we have data on year purchased house.


## THIS CALCS THE ALTERNATE ESTIMATE OF PROBABILITY OF ENTRY INTO HOUSING - NOT USED IN BASE CASE
## calc proportion of people in age and inc group who had not yet purchased a home, who purchase in current year
hilda_home_trans <- hilda_home_purchases %>%
  ## remove those who have previously purchased a home
  filter(first_home_purchase!=-1) %>%
  group_by(wavenumber, age_grp2, total_inc_qtile) %>%
  ## among non homeowners, how many purchased in given year
  summarise(home_trans_prob = sum(hhwte*first_home_purchase)/ sum(hhwte),
            sample_freq=n()) %>%
  ## average probs across all waves
  group_by(age_grp2, total_inc_qtile) %>%
  summarise(home_trans_prob = mean(home_trans_prob, na.rm=T),
            sample_freq=sum(sample_freq))

# hilda_home_trans_plot <- ggplot(hilda_home_trans) +
#   geom_col(aes(x = age_grp2, y = home_trans_prob)) +
#   facet_wrap(vars(total_inc_qtile))


## apply some smoothing over age groups
hilda_home_trans_smooth <- hilda_home_trans %>%
  group_by(total_inc_qtile) %>%
  arrange(total_inc_qtile, age_grp2) %>%
  mutate(across(contains("home_trans_prob"), ~custom_smoother(.x)))

# hilda_home_trans_smooth_plot <- ggplot(hilda_home_trans_smooth) +
#   geom_col(aes(x = age_grp2, y = home_trans_prob)) +
#   facet_wrap(vars(total_inc_qtile))

## ensure we have a probability for every age and inc combination
hilda_home_trans_smooth_complete <- expand_grid(age_grp2=unique(hilda_grouped$age_grp2),
                                                total_inc_qtile = unique(hilda_grouped$total_inc_qtile)) %>%
  left_join(hilda_home_trans_smooth %>% select(-sample_freq)) %>%
  mutate(home_trans_prob = ifelse(age_grp2=="[0,15)", 0, home_trans_prob)) %>%
  arrange(total_inc_qtile, age_grp2) %>%
  group_by(total_inc_qtile) %>%
  fill(home_trans_prob, .direction="downup")


## save
qsave(hilda_home_trans_smooth_complete, "./Input data/housing_transition_probs_ai_alt.qs")




# Individual new house price --------

## "starting" individual house prices in 2018. This will be grown with AWE.
## ie people who transition to homeowner in the model are assumed to buy a place with that price.
## Individual means the share attributed to the individual. Actual house price could be double for example if buying in a couple.


awe_projection <- qread("./Input data/AWE_projections.qs") %>% 
  mutate(wavenumber = year-2000)


##  av indiv house price in 2018 dollars by total inc and wave (no clear patterns observed by age grp)
av_houseprice_2018_inc_wave <- hilda_home_purchases %>% 
  filter(first_home_purchase==1 ) %>% 
  ## remove large vals of indiv home price - 16 of 4256 obs - main effect is to dampen quintile 1 value
  filter(psprice<1000000) %>% 
  group_by(wavenumber, total_inc_qtile) %>% 
  summarise(av_houseprice = sum(psprice*hhwte)/sum(hhwte),
            sample_freq = n()) %>% 
  ungroup() %>% 
  ## deflate by awe - deflated vals look reasonably straight over time
  left_join(awe_projection) %>% 
  mutate(av_houseprice_2018 = av_houseprice/awe_growth_factor_2018)

# ggplot(av_houseprice_2018_inc_wave) +
#   geom_col(aes(x = wavenumber, y=av_houseprice_2018, fill=total_inc_qtile), position="dodge")


## calc av across waves
av_houseprice_2018_inc <- av_houseprice_2018_inc_wave %>% 
  group_by(total_inc_qtile) %>% 
  summarise(av_houseprice_2018 = mean(av_houseprice_2018))

# ggplot(av_houseprice_2018_inc) +
#   geom_col(aes(x = total_inc_qtile, y=av_houseprice_2018), position="dodge")


## grow with AWE over the projection period
newhome_projection <- expand_grid(year=c(2018:2050), total_inc_qtile = av_houseprice_2018_inc$total_inc_qtile) %>% 
  left_join(av_houseprice_2018_inc %>% 
              mutate(year=2018) %>% 
              select(year, total_inc_qtile, newhome_price=av_houseprice_2018)) %>% 
  left_join(awe_projection %>% select(year, awe_growth_factor_2018)) %>% 
  group_by(total_inc_qtile) %>% 
  mutate(newhome_price = ifelse(is.na(newhome_price), newhome_price[1]*awe_growth_factor_2018 , newhome_price) )


# ggplot(newhome_projection) +
#   geom_line(aes(x = year, y=newhome_price, colour=total_inc_qtile))

## USED IN APPENDIX
## "These estimates ranged from about $252 000 to $332 000 in 2018 dollars for individuals across the five income groups."
newhome_projection %>% filter(year==2018)


## save
qsave(newhome_projection, "./Input data/newhousing_value_projection_i.qs")



# Given you are in housing, rate of net additional saving into housing via home purchases -------

## this captures both upsizing and downsizing
housing_change <- hilda_home_purchases %>% 
  ## indiv_home_prop is available currently for home purchase years
  ## get indiv_home_props for wealth years in addition to home purchase years
  mutate(indiv_home_prop = case_when(
    is.na(indiv_home_prop) & pwhmvai!=0 ~  pwhmvai/hwhmvai, ## this will calc for wealth years
    pwhmvai==0 ~ 0,
    TRUE ~ indiv_home_prop)) %>% 
  group_by(xwaveid) %>% 
  arrange(xwaveid, wavenumber) %>% 
  ## fill indiv_home_prop downwards and upwards - assume this share of home value attributable to all waves until new housing purchase or wealth year
  fill(indiv_home_prop, .direction="downup") %>% 
  ungroup %>% 
  ## missing indiv_home_prop - throw them out assuming they are randomly missing
  filter(!is.na(indiv_home_prop)) %>% 
  ## estimate value of housing assets attributable to individual - hsvalui available in each year
  mutate(housing_assets_indiv = hsvalui*indiv_home_prop) %>% 
  ## previous year's housing value 
  mutate(prev_housing_assets_indiv = ifelse(xwaveid == lag(xwaveid) & wavenumber == 1+lag(wavenumber),
                                            lag(housing_assets_indiv),
                                            NA)) %>% 
  ## estimate previous year's housing value if missing - assuming a rate of return of 7.2% (20 year average from Vanguard)
  mutate(last_record_housing_assets_indiv_est = case_when(
    xwaveid == lag(xwaveid) & wavenumber != 1+lag(wavenumber) ~ lag(housing_assets_indiv)
  ),
  last_record_wavenumber = case_when(
    xwaveid == lag(xwaveid) & wavenumber != 1+lag(wavenumber) ~ lag(wavenumber)
  ),
  prev_housing_assets_indiv_est = last_record_housing_assets_indiv_est*1.072^(wavenumber-last_record_wavenumber)
  ) %>% 
  mutate(prev_housing_assets_indiv = ifelse(is.na(prev_housing_assets_indiv), 
                                            prev_housing_assets_indiv_est, 
                                            prev_housing_assets_indiv)) %>% 
  ## filter to people who already owned a home
  filter(first_home_purchase==-1) %>% ## -1 means already owned a home
  ## in years purchased a new home, calculate change in housing value from previous year
  mutate(housing_change = case_when( home_purchased==1 & !is.na(prev_housing_assets_indiv) ~ psprice - prev_housing_assets_indiv,
                                     is.na(home_purchased) ~ 0 ## if didn't purchase a home that year, housing change is 0
  ))

## how many still missing prev housing value? about 10%. implicitly treated as zero change below
# temp <- housing_change %>% filter(home_purchased==1 & is.na(prev_housing_assets_indiv))
# temps <- housing_change %>% filter(home_purchased==1 & !is.na(housing_change))

## calculate change in housing value over existing housing value, by age
upsize_housing <- housing_change %>% 
  group_by(wavenumber, age_grp2) %>% 
  summarise(upsize_rate_housing = sum(housing_change*hhwte, na.rm=T)/sum(prev_housing_assets_indiv*hhwte, na.rm=T),
            sample_freq=n()) %>% 
  ## average across waves
  group_by(age_grp2) %>% 
  summarise(upsize_rate_housing = mean(upsize_rate_housing, na.rm=T),
            sample_freq=sum(sample_freq)) %>% 
  mutate(upsize_rate_housing = case_when(
    ## make 15-20 year olds 0 - unlikely to be upsizing from existing home
    age_grp2=="[15,20)"~ 0 , 
    ## make 85-105 same as prev age group - the increase was maybe due to sample
    age_grp2=="[85,105]" ~ lag(upsize_rate_housing),
    TRUE ~ upsize_rate_housing))

# ggplot(upsize_housing) +
#   geom_col(aes(x=age_grp2, y=upsize_rate_housing), position="dodge") 


## USED IN APPENDIX
## People aged under 70 tended to be net upsizers, with homeowners in the 30–34 age group estimated to increase their housing assets 
## by close to 5 per cent annually due to new home purchases, given historical rates of return.
## People aged 70 and over tended to be net downsizers, with rates of downsizing at less than 0.5 per cent. 
upsize_housing


qsave(upsize_housing, "./Input data/saving_housing_a.qs")



# Housing debt to asset ratio on first purchase ---------------------------------------------

## ie when people move to being a homeowner, this is the amount of housing debt they have as a % of housing assets
## with every "upsize/downsize", assume debt changes by the same ratio with the upsize/downsize

debt_to_asset_housing <- hilda_home_purchases %>%
  ## identify housing value and debt for first home purchase
  group_by(xwaveid) %>% 
  mutate(any_first_home_purchase = ifelse(first_home_purchase==1, 1, NA)) %>% 
  fill(any_first_home_purchase, .direction="downup") %>% 
  ungroup() %>% 
  filter(any_first_home_purchase==1) %>% 
  arrange(xwaveid, wavenumber) %>% 
  ## fill upwards housing assets and housing debt (available only in wealth years) - to be used in cases where hsdebti is 0 
  group_by(xwaveid) %>% 
  fill(housing_assets, housing_debt, .direction="up") %>% 
  ## get the housing value and debt in the wave after first home purchase (in case purchase not yet reflected in home value variable because purchased after surveyed)
  ##  also get housing value and debt in same wave of first home purchase, then pick which debt is higher
  mutate(housing_asset_after_purchase = ifelse(xwaveid==lead(xwaveid) & first_home_purchase==1, lead(hsvalui), NA ),
         housing_debt_after_purchase = ifelse(xwaveid==lead(xwaveid) & first_home_purchase==1, lead(hsdebti), NA ),
         housing_asset_at_purchase = ifelse(first_home_purchase==1, hsvalui, NA),
         housing_debt_at_purchase = ifelse(first_home_purchase==1, hsdebti, NA),
         
         housing_debt_to_use = ifelse(first_home_purchase==1 & housing_debt_at_purchase>=housing_debt_after_purchase, 
                                      housing_debt_at_purchase,
                                      housing_debt_after_purchase),
         housing_asset_to_use = ifelse(first_home_purchase==1 & housing_debt_at_purchase>=housing_debt_after_purchase, 
                                       housing_asset_at_purchase,
                                       housing_asset_after_purchase),
         
         ## if hsdebti is zero, may have been turned into an investment property. get values from wealth years where available as an approx
         housing_asset_to_use = ifelse(housing_debt_to_use==0, housing_assets, housing_asset_to_use),
         housing_debt_to_use = ifelse(housing_debt_to_use==0, housing_debt, housing_debt_to_use)) %>% 
  ## remove non first home purchases and strange obs
  filter(!is.na(housing_asset_to_use) & !is.na(housing_debt_to_use) & housing_asset_to_use!=0) %>% 
  group_by(wavenumber, age_grp2, total_inc_qtile) %>% 
  ## calc debt to asset ratio by age and inc
  summarise(housing_debt_to_asset = sum(housing_debt_to_use*hhwte, na.rm=T)/sum(housing_asset_to_use*hhwte, na.rm=T) ) %>% 
  ## average across waves
  group_by(age_grp2,  total_inc_qtile ) %>% 
  summarise(housing_debt_to_asset = mean(housing_debt_to_asset, na.rm=T)) %>% 
  ## smooth
  group_by(total_inc_qtile) %>% 
  arrange(total_inc_qtile, age_grp2) %>% 
  mutate(housing_debt_to_asset_smooth = custom_smoother(housing_debt_to_asset))

# ggplot(debt_to_asset_housing) +
#   geom_col(aes(x=age_grp2, y=housing_debt_to_asset_smooth)) +
#   facet_wrap(vars(total_inc_qtile))


## USED IN APPENDIX
## These estimates tended to be largest for younger age groups and decreased for older first home buyers. 
## in the third income quintile, the estimated debt to asset ratio was about 75 per cent for the 20–24 age group, 
## and about 40 per cent for the 50–54 age group. 
debt_to_asset_housing %>% filter(total_inc_qtile==3) %>% select(age_grp2, housing_debt_to_asset_smooth)


qsave(debt_to_asset_housing, "./Input data/newhousing_debt_ratio_ai.qs")


# Rate of housing debt reduction --------------------------------------------------

## estimate of how much debt reduces by (%) - mortgage repayments as a percentage of home loan size.
## increases due to new purchases are taken into account thru debt:asset ratio above

## mortgage repayments and complete paydowns as a percentage of home loan size...
debt_reduction_housing <- hilda_home_purchases %>% 
  mutate(homeowner = ifelse(first_home_purchase %in% c(-1, 1), 1, 0)) %>% ## -1 = already owned a home that year. 1 = bought a first home this year. 0 = havent bought a home
  filter(homeowner == 1) %>% 
  arrange(xwaveid, wavenumber) %>% 
  ## complete paydowns 
  mutate(complete_paydown = ifelse(xwaveid==lag(xwaveid) & wavenumber==1+lag(wavenumber) & hsdebti==0, lag(hsdebti), NA)) %>% 
  ## yearly mortgage repayments and complete paydowns. Limit complete paydowns to older age groups - more likely to be actual complete paydowns
  mutate(debt_reduction = case_when(
    xwaveid==lag(xwaveid) & wavenumber==1+lag(wavenumber) & hsdebti==0 & age_grp2>="[60,65)" ~ complete_paydown,
    hsmgi>=0 ~ hsmgi*12 ## est yearly mortgage repayments
  )) %>% 
  mutate(debt_denominator = case_when(
    xwaveid==lag(xwaveid) & wavenumber==1+lag(wavenumber) & !is.na(lag(hsdebti)) ~ lag(hsdebti),
    TRUE ~ hsdebti
  )) %>% 
  ## filter to people who had at least some debt in previous period
  mutate(had_debt = ifelse(hsdebti>0 | (xwaveid==lag(xwaveid) & wavenumber==1+lag(wavenumber) & lag(hsdebti>0)), 1, 0)) %>% 
  filter(had_debt == 1) %>% 
  ## debt reduction %
  group_by(wavenumber, age_grp2) %>% 
  summarise(debt_reduction = sum(debt_reduction*hhwte, na.rm=T) / sum(debt_denominator*hhwte, na.rm=T)) %>% 
  ## average over waves
  group_by(age_grp2) %>% 
  summarise(debt_reduction = mean(debt_reduction, na.rm=T))

# ggplot(debt_reduction_housing) +
#   geom_col(aes(x = age_grp2, y = debt_reduction)) 

## USED IN APPENDIX
## These estimates were relatively low for younger age groups, at about 8 per cent of housing debt,
## and increased at age groups older than 60, to about a third of housing debt.
debt_reduction_housing


## save
qsave(debt_reduction_housing, "./Input data/debt_reduction_housing_a.qs")